import AppLayout from '@/layouts/app-layout';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Alert, AlertDescription } from '@/components/ui/alert';
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from '@/components/ui/dialog';
import { Link, Head, router } from '@inertiajs/react';
import { useState } from 'react';
import { 
  Phone, 
  PhoneCall, 
  MessageSquare, 
  Search,
  Clock,
  CheckCircle,
  XCircle,
  Ban,
  AlertCircle
} from 'lucide-react';

interface NumberRequest {
  id: number;
  phone_number: {
    number: string;
    formatted_number: string;
    friendly_name: string;
    country_code: string;
    area_code: string;
    capabilities: {
      voice: boolean;
      sms: boolean;
      mms: boolean;
    };
    monthly_cost: number;
  };
  status: 'pending' | 'approved' | 'rejected' | 'cancelled';
  customer_notes: string | null;
  admin_notes: string | null;
  requested_at: string;
  processed_at: string | null;
}

interface PaginatedRequests {
  data: NumberRequest[];
  current_page: number;
  last_page: number;
  per_page: number;
  total: number;
}

interface Props {
  requests: PaginatedRequests;
}

export default function MyRequests({ requests }: Props) {
  const [selectedRequest, setSelectedRequest] = useState<NumberRequest | null>(null);
  const [isCancelModalOpen, setIsCancelModalOpen] = useState(false);
  const [isProcessing, setIsProcessing] = useState(false);

  const handleCancelRequest = (request: NumberRequest) => {
    setSelectedRequest(request);
    setIsCancelModalOpen(true);
  };

  const confirmCancel = () => {
    if (!selectedRequest) return;
    
    setIsProcessing(true);
    router.post(
      `/numbers/requests/${selectedRequest.id}/cancel`,
      {},
      {
        onSuccess: () => {
          setIsCancelModalOpen(false);
          setSelectedRequest(null);
        },
        onFinish: () => setIsProcessing(false),
      }
    );
  };

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'pending': return <Clock className="h-4 w-4" />;
      case 'approved': return <CheckCircle className="h-4 w-4" />;
      case 'rejected': return <XCircle className="h-4 w-4" />;
      case 'cancelled': return <Ban className="h-4 w-4" />;
      default: return <AlertCircle className="h-4 w-4" />;
    }
  };

  const getStatusVariant = (status: string): "default" | "secondary" | "destructive" | "outline" => {
    switch (status) {
      case 'pending': return 'secondary';
      case 'approved': return 'default';
      case 'rejected': return 'destructive';
      case 'cancelled': return 'outline';
      default: return 'outline';
    }
  };

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat('en-US', {
      style: 'currency',
      currency: 'USD',
    }).format(amount);
  };

  const formatDate = (date: string) => {
    return new Date(date).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'long',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit',
    });
  };

  const pendingCount = requests.data.filter(r => r.status === 'pending').length;
  const approvedCount = requests.data.filter(r => r.status === 'approved').length;
  const rejectedCount = requests.data.filter(r => r.status === 'rejected').length;

  return (
    <AppLayout>
      <Head title="My Requests" />
      
      <div className="space-y-6">
        {/* Header */}
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold tracking-tight">My Number Requests</h1>
            <p className="text-muted-foreground">
              Track the status of your phone number requests
            </p>
          </div>
          <div className="flex gap-2">
            <Button variant="outline" asChild>
              <Link href="/numbers/my-numbers">
                <Phone className="mr-2 h-4 w-4" />
                My Numbers
              </Link>
            </Button>
            <Button asChild>
              <Link href="/numbers/available">
                <Search className="mr-2 h-4 w-4" />
                Browse Available
              </Link>
            </Button>
          </div>
        </div>

        {/* Stats Cards */}
        {requests.data.length > 0 && (
          <div className="grid gap-4 md:grid-cols-4">
            <Card>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">Total Requests</CardTitle>
                <Phone className="h-4 w-4 text-muted-foreground" />
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold">{requests.total}</div>
                <p className="text-xs text-muted-foreground">
                  All time requests
                </p>
              </CardContent>
            </Card>

            <Card>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">Pending</CardTitle>
                <Clock className="h-4 w-4 text-yellow-600" />
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold">{pendingCount}</div>
                <p className="text-xs text-muted-foreground">
                  Awaiting review
                </p>
              </CardContent>
            </Card>

            <Card>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">Approved</CardTitle>
                <CheckCircle className="h-4 w-4 text-green-600" />
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold">{approvedCount}</div>
                <p className="text-xs text-muted-foreground">
                  Numbers assigned
                </p>
              </CardContent>
            </Card>

            <Card>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">Rejected</CardTitle>
                <XCircle className="h-4 w-4 text-red-600" />
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold">{rejectedCount}</div>
                <p className="text-xs text-muted-foreground">
                  Not approved
                </p>
              </CardContent>
            </Card>
          </div>
        )}

        {/* Requests List */}
        <Card>
          <CardHeader>
            <CardTitle>Request History</CardTitle>
            <CardDescription>
              View all your phone number requests and their current status
            </CardDescription>
          </CardHeader>
          <CardContent>
            {requests.data.length === 0 ? (
              <div className="text-center py-12">
                <Clock className="mx-auto h-12 w-12 text-muted-foreground mb-4" />
                <h3 className="text-lg font-semibold mb-2">No Requests Yet</h3>
                <p className="text-muted-foreground mb-6">
                  You haven't submitted any phone number requests. Browse available numbers to get started.
                </p>
                <Button asChild>
                  <Link href="/numbers/available">
                    <Search className="mr-2 h-4 w-4" />
                    Browse Available Numbers
                  </Link>
                </Button>
              </div>
            ) : (
              <>
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>Phone Number</TableHead>
                      <TableHead>Capabilities</TableHead>
                      <TableHead>Monthly Cost</TableHead>
                      <TableHead>Requested</TableHead>
                      <TableHead>Status</TableHead>
                      <TableHead>Notes</TableHead>
                      <TableHead className="text-right">Action</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {requests.data.map((request) => (
                      <TableRow key={request.id}>
                        <TableCell>
                          <div className="flex flex-col">
                            <span className="font-medium">{request.phone_number.formatted_number}</span>
                            <span className="text-sm text-muted-foreground">
                              {request.phone_number.country_code} • {request.phone_number.area_code}
                            </span>
                          </div>
                        </TableCell>
                        <TableCell>
                          <div className="flex gap-1">
                            {request.phone_number.capabilities.voice && (
                              <Badge variant="secondary" className="text-xs">
                                <PhoneCall className="mr-1 h-3 w-3" />
                                Voice
                              </Badge>
                            )}
                            {request.phone_number.capabilities.sms && (
                              <Badge variant="secondary" className="text-xs">
                                <MessageSquare className="mr-1 h-3 w-3" />
                                SMS
                              </Badge>
                            )}
                          </div>
                        </TableCell>
                        <TableCell className="font-medium">
                          {formatCurrency(request.phone_number.monthly_cost)}
                        </TableCell>
                        <TableCell className="text-sm text-muted-foreground">
                          {formatDate(request.requested_at)}
                        </TableCell>
                        <TableCell>
                          <Badge variant={getStatusVariant(request.status)}>
                            <span className="flex items-center gap-1">
                              {getStatusIcon(request.status)}
                              {request.status.charAt(0).toUpperCase() + request.status.slice(1)}
                            </span>
                          </Badge>
                        </TableCell>
                        <TableCell>
                          {request.status === 'rejected' && request.admin_notes ? (
                            <div className="max-w-xs">
                              <p className="text-sm text-red-600 font-medium">Rejection Reason:</p>
                              <p className="text-sm text-muted-foreground truncate">{request.admin_notes}</p>
                            </div>
                          ) : request.status === 'approved' && request.admin_notes ? (
                            <div className="max-w-xs">
                              <p className="text-sm text-muted-foreground truncate">{request.admin_notes}</p>
                            </div>
                          ) : request.customer_notes ? (
                            <p className="text-sm text-muted-foreground truncate max-w-xs">{request.customer_notes}</p>
                          ) : (
                            <span className="text-sm text-muted-foreground">—</span>
                          )}
                        </TableCell>
                        <TableCell className="text-right">
                          {request.status === 'pending' ? (
                            <Button
                              variant="outline"
                              size="sm"
                              onClick={() => handleCancelRequest(request)}
                            >
                              Cancel
                            </Button>
                          ) : request.status === 'approved' ? (
                            <Button variant="ghost" size="sm" asChild>
                              <Link href="/numbers/my-numbers">View Number</Link>
                            </Button>
                          ) : (
                            <span className="text-sm text-muted-foreground">—</span>
                          )}
                        </TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>

                {/* Pagination */}
                {requests.last_page > 1 && (
                  <div className="flex items-center justify-between pt-4">
                    <div className="text-sm text-muted-foreground">
                      Showing {requests.data.length} of {requests.total} requests
                    </div>
                    <div className="flex gap-2">
                      <Button
                        variant="outline"
                        size="sm"
                        disabled={requests.current_page === 1}
                      >
                        Previous
                      </Button>
                      <Button
                        variant="outline"
                        size="sm"
                        disabled={requests.current_page === requests.last_page}
                      >
                        Next
                      </Button>
                    </div>
                  </div>
                )}

                {/* Help Alert */}
                {pendingCount > 0 && (
                  <Alert className="mt-4">
                    <Clock className="h-4 w-4" />
                    <AlertDescription>
                      You have {pendingCount} pending {pendingCount === 1 ? 'request' : 'requests'}. An admin will review and respond soon. You'll receive an email notification when your request is processed.
                    </AlertDescription>
                  </Alert>
                )}
              </>
            )}
          </CardContent>
        </Card>
      </div>

      {/* Cancel Confirmation Modal */}
      <Dialog open={isCancelModalOpen} onOpenChange={setIsCancelModalOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Cancel Request</DialogTitle>
            <DialogDescription>
              Are you sure you want to cancel this phone number request?
            </DialogDescription>
          </DialogHeader>
          
          {selectedRequest && (
            <div className="py-4">
              <div className="rounded-lg border p-4 space-y-2">
                <div className="flex items-center justify-between">
                  <span className="text-sm font-medium">Phone Number:</span>
                  <span className="text-lg font-bold">{selectedRequest.phone_number.formatted_number}</span>
                </div>
                <div className="flex items-center justify-between">
                  <span className="text-sm font-medium">Requested:</span>
                  <span className="text-sm text-muted-foreground">{formatDate(selectedRequest.requested_at)}</span>
                </div>
              </div>

              <Alert className="mt-4" variant="destructive">
                <AlertCircle className="h-4 w-4" />
                <AlertDescription>
                  This action cannot be undone. The number will be returned to the available pool and you'll need to submit a new request if you change your mind.
                </AlertDescription>
              </Alert>
            </div>
          )}

          <DialogFooter>
            <Button
              variant="outline"
              onClick={() => setIsCancelModalOpen(false)}
              disabled={isProcessing}
            >
              Keep Request
            </Button>
            <Button
              variant="destructive"
              onClick={confirmCancel}
              disabled={isProcessing}
            >
              {isProcessing ? 'Cancelling...' : 'Cancel Request'}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </AppLayout>
  );
}
